/*=================================================================
 *
 * interp_and_exp_transform_c.c ... Similar to ppval in Matlab, but faster since optimized for cubic splines AND equally spaced data
 *
 * The calling syntax is:
 *
 *		interp_and_exp_transform_c.c(xx, b, c, index)
 *
 *  dz      ... spacing of points where to evaluate cubic spline
 *  N       ... number of observations: 0, dz, 2*dz, ..., (N-1)*dz
 *  b       ... support points of spline
 *  c       ... spline coefficients
 *  index   ... matching coefficient rows for points xx
 *
 * Sample call:
 *
 *      interp_and_exp_transform_c(0:0.1:1, [0.2 0.8], [1 1 1 1; 2 2 2 2])
 *
 *=================================================================*/

#include <math.h>
#include "mex.h"

#ifndef max
#define max(a,b) (((a) > (b)) ? (a) : (b))
#endif

/* Compute histogram bins */
static void histc_sorted_c2(int bins[], double dz[], double edges[], int num_points_int, unsigned int num_edges)
{
    int i, j;
    
    i = j = 0;
    while (i < num_points_int)
    {
        while ((dz[0] * i > edges[j] - 3e-11) && (j < num_edges))
        {
        	j++;
        }
        bins[i] = j;
        i++;
    }
    
    return;
}

/* Evaluate cubic spline at xx */
static void evaluate_cubic_spline_c2(double dz[], double yy[], double b[], double c[], int index[], int num_points_int, unsigned int num_breaks)
{
    int i, k, c_length, tmp, tmp2;
    double xx_local2;
    
    /* Initialize y */
    /*for (i = 0; i<num_points_int; i++) {
        yy[i] = c[index[i]-1];
    }*/
    
    /* Apply nested multiplication */
    c_length = num_breaks - 1;
    for (k = 1; k<4; k++) {        
        tmp = c_length * k - 1;
        for (i = 0; i<num_points_int; i++) {
            /* Go to local coordinates */
            tmp2 = index[i]-1;
            xx_local2 = dz[0]*i - b[tmp2];
            if (k==1) {                                
                yy[i] = c[tmp2];
            }
            
            /* Evaluate spline */
            yy[i] = xx_local2 * yy[i] + c[index[i] + tmp];
        }
    }

    return;
}


/* Gateway routine (to Matlab) */
void mexFunction( int nlhs, mxArray *plhs[], 
		  int nrhs, const mxArray*prhs[] )
     
{ 
    int i;
    double *xx, *yy_real, *yy_imag, *b_real, *c_real, *b_imag, *c_imag, tmp_b0, *dz, *num_points, *transform_real, *transform_imag, tmp; 
    unsigned int num_points1, num_points2, num_breaks;
    int num_points_int;
    int *index_real, *index_imag;
    const int *dims;
    
    /* Get number of data points */
    dz = mxGetPr(prhs[0]);
    num_points = mxGetPr(prhs[1]);
    num_points_int = (int) num_points[0];
    
    num_points1 = mxGetM(prhs[2]);
    num_points2 = mxGetN(prhs[2]);
    num_breaks = max(num_points1, num_points2);
    
    /* Allocate memory */
    plhs[0] = mxCreateDoubleMatrix(1, num_points_int, mxCOMPLEX);
    transform_real = mxGetPr(plhs[0]);
    transform_imag = mxGetPi(plhs[0]);
    if (true) {
        /* Allocate memory, faster than Matlab version below */
        index_real = malloc (sizeof (i) * num_points_int);
        index_imag = malloc (sizeof (i) * num_points_int);
        yy_real = malloc (sizeof (tmp_b0) * num_points_int);
        yy_imag = malloc (sizeof (tmp_b0) * num_points_int);
    } else {
        /*xx_local =  mxGetPr(mxCreateDoubleMatrix(num_points_int, 1, mxREAL));
        dims = mxGetDimensions(prhs[0]);
        index = mxGetPr(mxCreateNumericArray(2, dims, mxINT32_CLASS, 0)); */
    }
    
    /* Assign pointers to the various parameters */ 
    b_real = mxGetPr(prhs[2]);
    c_real = mxGetPr(prhs[3]);
    b_imag = mxGetPr(prhs[4]);
    c_imag = mxGetPr(prhs[5]);
    
    /* For each evaluation site, compute its breakpoint interval */
    tmp_b0 = b_real[0];
    b_real[0] = -1e20;
    histc_sorted_c2(index_real, dz, b_real, num_points_int, num_breaks);
    b_real[0] = tmp_b0;
    
    tmp_b0 = b_imag[0];
    b_imag[0] = -1e20;
    histc_sorted_c2(index_imag, dz, b_imag, num_points_int, num_breaks);
    b_imag[0] = tmp_b0;
    
    /* Go to local coordinates and apply nested multiplication */
    evaluate_cubic_spline_c2(dz, yy_real, b_real, c_real, index_real, num_points_int, num_breaks);
    evaluate_cubic_spline_c2(dz, yy_imag, b_imag, c_imag, index_imag, num_points_int, num_breaks);
    
    /* Put real and imaginary part of log-transform together and take exponent */
    for (i=0; i<num_points_int; i++) {
        tmp = exp(yy_real[i]);
        transform_real[i] = tmp * cos(yy_imag[i]);
        transform_imag[i] = tmp * sin(yy_imag[i]);
    }
    
    /* Free up allocated memory */
    if (true) {
        free(index_real);
        free(index_imag);
        free(yy_real);
        free(yy_imag);
    }
    //transform_imag[0] = num_breaks;
    return;
}
